<?php
/* Copyright (C) 2017 Sergi Rodrigues <proyectos@imasdeweb.com>
 *
 * Licensed under the GNU GPL v3 or higher (See file gpl-3.0.html)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

/**
 *	\file       htdocs/stockTransfers/lib/stocktransfers.lib.php
 *      \ingroup    stockTransfers
 *      \brief      Mix of functions for Import bank movements module
 *      \version    v 1.0 2017/11/20
 */

$linktohelp='EN:Module_stockTransfers_En|CA:Modul_stockTransfers|ES:Modulo_stockTransfers';

/*
	these are to guarantee compatibility to Dolibarr versions previous to 10 and 11, where didn't exist newToken() and currentToken()
*/
if (!function_exists('newToken')) {
	/**
	 * returns the NEW token generated by Dolibarr (i created this function for compatibility with old Dolibarr versions)
	 *
	 * @return	string
	 */
	function newToken()
	{
		return empty($_SESSION['newtoken']) ? '' : $_SESSION['newtoken'];
	}
}
if (!function_exists('currentToken')) {
	/**
	 * returns the CURRENT token generated by Dolibarr (i created this function for compatibility with old Dolibarr versions)
	 *
	 * @return	string
	 */
	function currentToken()
	{
		return isset($_SESSION['token']) ? $_SESSION['token'] : '';
	}
}

/**
 * Returns a formatted version of a price
 * note: this function should not be necessary, but i've not understood why the dolibarr price() function doesn't render thousands separator
 *
 * @param	decimal		$floatval	Floatval without format
 * @return	string					Formated price
 */
function _price($floatval)
{
	global $langs, $db, $conf;
	$dec = $langs->transnoentitiesnoconv("SeparatorDecimal");
	if ($dec==',')
		return number_format(floatval($floatval), 2, ',', '.');
	else return number_format(floatval($floatval), 2, '.', ',');
}

/**
 * Returns a formatted version of a quantity (optionally with decimal digits, depending on module settings)
 *
 * @param	Decimal		$floatval	Floatval without format
 * @return	String					Formated quantity
 */
function _qty($floatval)
{
	global $langs, $db, $conf;
	$dec = $langs->transnoentitiesnoconv("SeparatorDecimal");
	$num_decimals = $value = !empty($conf->global->STOCKTRANSFERS_MODULE_SETT_14) ? intval($conf->global->STOCKTRANSFERS_MODULE_SETT_14) : 0;
	if ($dec==',')
		return number_format(floatval($floatval), $num_decimals, ',', '.');
	else return number_format(floatval($floatval), $num_decimals, '.', ',');
}

/**
 * Load the rendering of a template view
 *
 * @param	String	$viewname	string
 * @param	Array	$vars		array of vars to be shared with the PHP view file
 * @return	String
 */
function STRANS_render_view($viewname, Array $vars)
{
	global $langs, $db, $conf, $user;

	// passed vars
	if (count($vars)>0) {
		foreach ($vars as $__k__=>$__v__) {
			${$__k__} = $__v__;
		}
	}

	// we save a copy of the content already existing at the output buffer (for no interrump it)
	$existing_render = ob_get_clean();

	// we begin a new output
	ob_start();
	include __DIR__.'/views/'.$viewname.'.php';

	// we get the current output
	$render = ob_get_clean();

	// we re-send to output buffer the existing content before to arrive to this function ;)
	ob_start();
	echo $existing_render;

	return $render;
}

/*
 * I must use this function_exists() because the same functions are used by all IMASDEWEB modules
 * and it avoid to re-define twice the same functions causing a PHP error
 */

if (!function_exists('_var')) {
	//
	/**
	 * Nicely render an array/object
	 *
	 * @param	array	$arr					Array/object to be rendered
	 * @param	string	$title					Optional title to head the render
	 * @param	integer	$b_htmlspecialchars		Optional, 1 = apply htmlspecialchars function
	 * @param	boolean	$is_object				Optional, recommended leave not defined
	 * @param	integer	$ii						Optional, recommended leave not defined
	 * @return	string
	 */
	function _var($arr, $title = '', $b_htmlspecialchars = 1, $is_object = '', $ii = 0)
	{
		$ii++;
		if ($ii==1) { // root DIV
			$html = "\n<div style='font-family:monospace;word-break:break-all;'>";
		} else {
			$html = "\n<div style='margin-left:100px;'>";
		}

		if (is_resource($arr)) {
			$arr = 'RESOURCE OF TYPE: '.get_resource_type($arr); // -> "convert" resource to string
			$is_object = false;
		} elseif (is_object($arr)) {
			$is_object = true;
			$arr = get_object_vars($arr);
		} elseif ($is_object=='') {
			$is_object = false;
		}

		$n_elements = 0;
		if (is_array($arr)) {
				$n_elements = count($arr);
			if ($n_elements==0) {
				$html .= "&nbsp;";
			} else {
				foreach ($arr as $k=>$ele) {
					$html .= "\n\t<div style='float:left;'><b style='".($is_object ? 'background-color:rgba(0,0,0,0.1);padding:2px':'')."'>$k <span class='arrow' style='color:#822;'>&rarr;</span> </b></div>";
					$html .= "\n\t\t<div style='border:1px #ddd solid;font-size:10px;font-family:sans-serif;'>";
					$html .= is_object($ele) || is_resource($ele)? _var(get_object_vars($ele), '', $b_htmlspecialchars, true, $ii) : _var($ele, '', $b_htmlspecialchars, false, $ii);
					$html .= "\n\t\t</div>";
					$html .= "\n\t<div style='float:none;clear:both;'></div>";
				}
			}
		} elseif ($arr===null) {
			$html .= "&nbsp;";
		} elseif (substr($arr, 0, 2)=='{"' || substr($arr, 0, 3)=='[{"') {
			$json = f_json_decode($arr);
			if (is_array($json)) {
				$n_elements = count($json);
				$html .=  htmlspecialchars($arr).'<br /><br />'._var($json, '', $b_htmlspecialchars, false, $ii).'<br />';
			} else {
				$html .=  $arr;
			}
		} elseif ($arr === 'b:0;' || substr($arr, 0, 2)=='a:') {
			$uns = f_unserialize($arr);
			if (is_array($uns)) {
				$n_elements = count($uns);
				$html .= htmlspecialchars($arr).'<br /><br />'._var($uns, '', $b_htmlspecialchars, false, $ii).'<br />';
			} else {
				$html .= $b_htmlspecialchars==1 ? htmlspecialchars($arr) : $arr;
			}
		} else {
			$n_elements = 1;
			$html .= $b_htmlspecialchars==1 ? htmlspecialchars($arr) : $arr;
		}
		$html .= "</div>";

		// title
		if (!empty($title)) {
			$html = '<h3>'.$title.' <span style="opacity:0.5;font-family:monospace;">['.$n_elements.']</span></h3>'.$html;
		}

		return $html;
	}

	/**
	 * Nicely render an array
	 *
	 * @param	$v	array/object
	 * @param	$t	string
	 * @return	void
	 */
	function _evar($v, $t = '')
	{
		echo _var($v, $t);
	}

	/**
	 * Nicely render an array and die
	 *
	 * @param	$v	array/object
	 * @param	$t	string
	 * @return	void
	 */
	function _evard($v, $t = '')
	{
		echo _var($v, $t); die();
	}
}

if (!function_exists('f_unserialize')) {

	/**
	 * Smart unserialize
	 *
	 * @param	$string					string
	 * @param	$return_always_an_array	boolean
	 * @return	string
	 */
	function f_unserialize($string, $return_always_an_array = true)
	{
		$string = trim($string);
		if (!is_string($string) || $string=='' || substr($string, 0, 2)!='a:') {
			if ($return_always_an_array)
				return array();
			else return $string;
		}
		$uns = @unserialize($string);
		if (!is_array($uns) || (count($uns)==0 && strlen($string)>'4')) {
			$string = f_fix_serialized($string);
			$uns2 = @unserialize($string);
			if (is_array($uns2)) {
				return $uns2;
			} else {
				return array();
			}
		} elseif (is_array($uns)) {
			return $uns;
		} else {
			return array();
		}
	}
}

if (!function_exists('f_json_decode')) {

	/**
	 * Smart json_decode
	 *
	 * @param	$string					string
	 * @param	$return_always_an_array	boolean
	 * @return	string
	 */
	function f_json_decode($string, $return_always_an_array = true)
	{
		$string = trim($string);
		if (!is_string($string) || $string==''
			|| (substr($string, 0, 1)!='{' && substr($string, 0, 1)!='[') ) {
			if ($return_always_an_array)
				return array();
			else return $string;
		}

		$uns = json_decode($string, true); // the second parameter (true) force the convertion of objects to associative arrays
		if (is_array($uns)) {
			return $uns;
		} else {
			if ($return_always_an_array)
				return array();
			else return $string;
		}
	}

	/**
	 * Fix unserialize: a:32:{i:1;s:12:"2010/02/22";i:2;... -> a:32:{i:1;s:10:"2010/02/22";i:2;...
	 *
	 * @param	$string	string
	 * @return	string
	 */
	function f_fix_serialized($string)
	{
		$reg = '/s:(\d+):"(.*?)";/';
		if (preg_match($reg, $string)) {
			$string = preg_replace_callback($reg, function ($matches) {
				return "s:".strlen($matches[2]).":\"".$matches[2]."\";"; }, $string);
		}
		return $string;
	}
}

/*
 * retrieve the translations in the available languages of this module of an array of KEY TEXTS
 */
if (!function_exists('f_multi_translation')) {
	/**
	 * Return an associative array by language for a set of serialized translated language keys
	 *
	 * @param	$a_keys		array
	 * @param	$languages	array
	 * @return	array associative of css classes for each file extension
	 */
	function f_multi_translation($a_keys, $languages)
	{

		$translations = array();
		foreach ($a_keys as $key) {
			$translations[$key] = array();
		}

		$languages = scandir(STOCKTRANSFERS_MODULE_DOCUMENT_ROOT.'/langs');
		foreach ($languages as $langcode) {
			if ($langcode=='.' || $langcode=='..') continue;
			$ex_lang = explode('_', $langcode);
			$lang = $ex_lang[0];
			$file_path = STOCKTRANSFERS_MODULE_DOCUMENT_ROOT.'/langs/'.$langcode.'/stocktransfers.lang';
			if (!file_exists($file_path) || !is_readable($file_path)) continue;
			$fp = @fopen(STOCKTRANSFERS_MODULE_DOCUMENT_ROOT.'/langs/'.$langcode.'/stocktransfers.lang', "r");
			if (!$fp) continue;
			while (($line = fgets($fp, 4096)) !== false) {
				if (trim($line)=='') continue;
				$ex = explode('=', $line);
				if (count($ex)<2) continue;
				if (isset($translations[$ex[0]])) {
					$translations[$ex[0]][$lang] = $ex[1];
				}
			}
		}

		return $translations;
	}
}

if (!function_exists('f_json_decode_translation')) {

	/**
	 * Return an associative array by language for a set of serialized translated language keys
	 *
	 * @param	String	$const			Name of the global constant (module setting) containing the serializated translations
	 * @param	String	$defaultLang	Default language
	 * @return	Array 					associative of css classes for each file extension
	 */
	function f_json_decode_translation($const, $defaultLang)
	{
		global $conf;

		$sett = !empty($conf->global->$const) ? $conf->global->$const : '';
		if (!preg_match('/\{/', $sett)) { // already NOT is a JSON format... so it's a recently updated module to version 1.23
			$s_translations = array();
			$s_translations[$defaultLang] = $sett;
		} else {
			$s_translations = json_decode($sett, JSON_OBJECT_AS_ARRAY);
		}
		return $s_translations;
	}
}

if (!function_exists('f_switch_control')) {

	/**
	 * Render an HTML/javascript switch control for forms
	 *
	 * @param	$vars		array ('name'=>'TICKETPLUS_MODULE_SETT_TIC_03','options'=>array('N','Y'),'value'=>'N')
	 * @return	string
	 */
	function f_switch_control($vars)
	{
		global $langs;

		$input_id = str_replace(array('[',']'), '_', $vars['name']).'___input';

		$onclick  = "\$.each($(this).find('em'),function(k,em){
						if(\$(em).is(':visible')){
							\$(em).hide();
						}else{
							\$(em).show();\$('#".$input_id."').val(\$(em).attr('v'));
						}
					});"
					.(!empty($vars['onclick']) ? str_replace('$', '\$', $vars['onclick']) : '');

		return "
				<input type='hidden' name='".$vars['name']."' id='".$input_id."' value='".$vars['value']."' />
				<span style='cursor:pointer;' onclick=\"".$onclick."\">
					<em v='".$vars['options'][0]."' style='".($vars['value']==$vars['options'][0]?"":"display:none;")."'>".img_picto($langs->trans("Disabled"), 'switch_off')."</em>
					<em v='".$vars['options'][1]."' style='".($vars['value']==$vars['options'][1]?"":"display:none;")."'>".img_picto($langs->trans("Disabled"), 'switch_on')."</em>
				</span>
		";
	}
}

if (!function_exists('f_query')) {
	/**
	 * run SQL query of SELECT type and return array of records
	 *
	 * @param	string	$sql			SQL query
	 * @param	string	$index_field	field name to be used to index the array of results
	 * @return	array
	 */
	function f_query($sql, $index_field = '')
	{
		global $db;

		$elements = array();
		$resql=$db->query($sql);
		if ($resql) {
			if ($db->num_rows($resql)) {
				while ($row = $db->fetch_array($resql)) {
					if (is_array($row)) {
						if (!empty($index_field) && isset($row[$index_field])) {
							$elements[$row[$index_field]] = $row;
						} else {
							$elements[] = $row;
						}
					}
				}
			}
			$db->free($resql);
			return $elements;
		} else {
			return "Error ".$db->lasterror();
			dol_syslog('@STSTST -- '.get_class($this)."::query ERROR: ".$db->lasterror(), LOG_ERR);
		}
	}
}
