<?php
/* Copyright (C) 2017 Sergi Rodrigues <proyectos@imasdeweb.com>
 *
 * Licensed under the GNU GPL v3 or higher (See file gpl-3.0.html)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

/**
 *	\file       htdocs/stocktransfers/core/boxes/box_stocktransfers.php
 *      \defgroup   stocktransfers Module Stock transfers
 *      \brief      Box for Dolibarr dashobard
 *      \version    v 1.0 2017/11/20
 */

dol_include_once('/core/boxes/modules_boxes.php');
dol_include_once('/stocktransfers/lib/stocktransfers.lib.php');

$langs->load("stocktransfers");


/**
 * Class to manage the box to show last stocktransfers
 */
class box_stocktransfers extends ModeleBoxes
{
	public $boxcode="laststocktransfers";
	public $boximg="object_products";
	public $boxlabel='stocktransfersBoxTitle';
	public $depends = array("stocktransfers");

	public $db;
	public $enabled = 1;

	public $info_box_head = array();
	public $info_box_contents = array();

	/**
	 *  Constructor
	 *
	 *  @param  DoliDB	$db      	Database handler
	 *  @param	string	$param		More parameters
	 */
	public function __construct($db, $param = '')
	{
		global $conf, $user;

		$this->db = $db;
		$this->hidden =  empty($user->rights->stock->mouvement->lire) || empty($user->rights->produit);
	}

	/**
	 *  Load data for box to show them later
	 *
	 *  @param	int		$max        Maximum number of records to load
	 *  @return	void
	 */
	public function loadBox($max = 5)
	{
		global $user, $langs, $db, $conf;
		$langs->load("boxes");

		$this->max = $max;

		// == check permissions
		if (empty($user->rights->produit) && empty($user->rights->fournisseur) && empty($user->rights->societe)) {
			$this->info_box_contents[0][0] = array(
				'td' => '',
				'text' => $langs->trans("ReadPermissionNotAllowed"),
			);
			return;
		}

		// == STOCKTRANSFERS_MODULE DOCUMENT_ROOT & URL_ROOT
		if (!defined('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT')) {
			if (file_exists(DOL_DOCUMENT_ROOT.'/custom/stocktransfers/core/modules/modStocktransfers.class.php')) {
				define('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT', DOL_DOCUMENT_ROOT.'/custom/stocktransfers');
				define('STOCKTRANSFERS_MODULE_URL_ROOT', DOL_URL_ROOT.'/custom/stocktransfers');
			} else {
				define('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT', DOL_DOCUMENT_ROOT.'/stocktransfers');
				define('STOCKTRANSFERS_MODULE_URL_ROOT', DOL_URL_ROOT.'/stocktransfers');
			}
		}

		// == load data
		include_once STOCKTRANSFERS_MODULE_DOCUMENT_ROOT.'/lib/stocktransfers_transfer.class.php';
		$transfer = new StockTransfer($db);
		$transfers = $transfer->getLatestTransfers(array('max'=>$max));

		// == load depots
		$depots = array();
		$resql = $db->query("SELECT * FROM ".MAIN_DB_PREFIX."entrepot");
		if ($resql) {
			while ($row = $db->fetch_array($resql)) $depots[$row['rowid']] = $row;
		}

		// == box head
		$textHead = $langs->trans("stocktransfersBoxTitle", $max);
		if (file_exists(DOL_DOCUMENT_ROOT.'/custom/stocktransfers/transfer_list.php')) {
			$sublink = 'custom/stocktransfers/transfer_list.php';
		} elseif (file_exists(DOL_DOCUMENT_ROOT.'/stocktransfers/transfer_list.php')) {
			$sublink = 'stocktransfers/transfer_list.php';
		} else {
			$sublink = '';
		}
		$this->info_box_head = array(
			'text' => $textHead,
			'limit'=> dol_strlen($textHead),
			'sublink'=>$sublink,
			'subtext'=>($sublink!='' ? $langs->trans("List"):''),
			'subpicto'=>($sublink!='' ? 'list':''),
		);

		// == render
		dol_syslog('@STSTST -- '.get_class($this)."::loadBox", LOG_DEBUG);
		$status_picto = array('0'=>'0','1'=>'1','2'=>'3','3'=>'4');
		$lines = 0;
		foreach ($transfers as $t) {
			$a_line = array();

			// = column: date and link to edit record
			$url = STOCKTRANSFERS_MODULE_URL_ROOT.'/transfer_edit.php?mainmenu=products&leftmenu=&rowid='.$t['rowid'];
			$label = !empty($t['date1']) ? $t['date1'] : '#'.$t['rowid'];
			$title = !empty($t['label']) ? $t['label'] : '';
			$text_link = " <a href='$url' title='".(str_replace("'", "", $title))."'><i class='fa fa-truck'></i> ".$label."</a>";

			$a_line[] = array(
				'td' => 'align="left"',
				'text' => $text_link,
				'asis' => 1,
			);

			// = column: depot 1
			$url = 'product/stock/card.php?id='.$t['fk_depot1'];
			$picto_link = "<a href='$url'><img src='theme/".$conf->theme."/img/object_company.png' /></a>";
			$text_link = " <a href='$url'>".(!empty($depots[$t['fk_depot1']]) ? $depots[$t['fk_depot1']]['ref'] : '#'.$t['fk_depot1'])."</a>";
			$a_line[] = array(
						'td' => 'align="left"',
						'text' => $picto_link.$text_link,
						'asis' => 1,
			);

			// = column: depot 2
			$url = 'product/stock/card.php?id='.$t['fk_depot2'];
			$picto_link = "<a href='$url'><img src='theme/".$conf->theme."/img/object_company.png' /></a>";
			$text_link = " <a href='$url'>".(!empty($depots[$t['fk_depot2']]) ? $depots[$t['fk_depot2']]['ref'] : '#'.$t['fk_depot2'])."</a>";
			$a_line[] = array(
						'td' => 'align="left"',
						'text' => $picto_link.$text_link,
						'asis' => 1,
			);

			// = column: number of products included
			$a_line[] = array(
						'td' => 'align="center"',
						'text' => $t['n_products']. ' '. $langs->trans('STProducts')
			);

			// = column: status
			$text = '';
			if (!isset($t['status']))
				$text = '';
			elseif ($t['status']=='0')
				$text = img_picto($langs->trans('stocktransfersStatus0'), 'statut0');
			elseif ($t['status']=='1')
				$text = img_picto($langs->trans('stocktransfersStatus1'), 'statut3');
			elseif ($t['status']=='2')
				$text = img_picto($langs->trans('stocktransfersStatus2'), 'statut4');

			$a_line[] = array(
						'td' => 'align="center" width="18"',
						'text' => $text
			);

			$this->info_box_contents[$lines] = $a_line;

			$lines++;
		}

		if ($lines==0) $this->info_box_contents[$lines][0] = array('td' => 'align="center"','text'=>$langs->trans("STNoTransferYet"));
	}

	/**
	 *	Method to show box
	 *
	 *	@param	array	$head       Array with properties of box title
	 *	@param  array	$contents   Array with properties of box lines
	 *  @param	int		$nooutput	No print, only return string
	 *	@return	string
	 */
	public function showBox($head = null, $contents = null, $nooutput = 0)
	{
		return parent::showBox($this->info_box_head, $this->info_box_contents, $nooutput);
	}
}
