<?php
/* Copyright (C) 2017 Sergi Rodrigues <proyectos@imasdeweb.com>
 *
 * Licensed under the GNU GPL v3 or higher (See file gpl-3.0.html)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

// Put here all includes required by your class file

require_once DOL_DOCUMENT_ROOT.'/core/class/commonobject.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/stock/class/productlot.class.php';

// == STOCKTRANSFERS_MODULE DOCUMENT_ROOT & URL_ROOT
if (!defined('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT')) {
	if (file_exists(DOL_DOCUMENT_ROOT.'/custom/stocktransfers/core/modules/modStocktransfers.class.php')) {
		define('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT', DOL_DOCUMENT_ROOT.'/custom/stocktransfers');
		define('STOCKTRANSFERS_MODULE_URL_ROOT', DOL_URL_ROOT.'/custom/stocktransfers');
	} else {
		define('STOCKTRANSFERS_MODULE_DOCUMENT_ROOT', DOL_DOCUMENT_ROOT.'/stocktransfers');
		define('STOCKTRANSFERS_MODULE_URL_ROOT', DOL_URL_ROOT.'/stocktransfers');
	}
}

/**
 * Functions needed to CRUD stocktransfers
 */
class StockTransfer extends CommonObject
{
	public $db;							//!< To store db handler
	public $error;							//!< To return error code (or message)
	public $errors=array();				//!< To return several error codes (or messages)

	public $element='transfer';
	public $table_element='stocktransfers_transfers';
	public $picto='stocktransfers';

	public $entity;
	public $rowid;
	public $ts_create;
	public $fk_depot1;
	public $fk_depot2;
	public $date1;
	public $date2;
	public $fk_user_author;
	public $fk_project;
	public $label;
	public $inventorycode;
	public $shipper;
	public $private_note;
	public $pdf_note;
	public $n_package = 1;
	public $status = '0'; // 0->draft, 1->validated-not-delivered, 2->delivered

	public $s_products = '';
	public $products = array();

	public $s;
	public $b_batch_enabled;

	const STATUS_DRAFT = 0;
	const STATUS_VALIDATED = 1;
	const STATUS_DELIVERED = 2;

	/**
	 * Class constructor
	 *
	 * @param DoliDB $DB Database handler
	 */
	public function __construct($DB)
	{
		global $conf;
		$this->db = $DB;
		$this->entity = $conf->entity;
		
		$this->b_batch_enabled = isModEnabled('productbatch')
									&& (empty($conf->global->STOCKTRANSFERS_MODULE_SETT_08)
											|| $conf->global->STOCKTRANSFERS_MODULE_SETT_08 != 'N') ? true : false;

		return 1;
	}

	/**
	 * Add a new stocktransfer at database
	 *
	 * @return array
	 */
	public function create()
	{
		global $conf, $langs, $user;
		$error=0;

		// Check parameters
		if (empty($user->id) || empty($this->fk_depot1) || empty($this->fk_depot2)) {
				$this->error = "ErrorBadParameter";
				dol_syslog('@STSTST -- '.get_class($this)."::create ERROR: Try to create a transfer with an empty parameter (user, depots, ...)", LOG_ERR);
				return -3;
		}

		// Insert request
		$sql = "INSERT INTO ".MAIN_DB_PREFIX."stocktransfers_transfers(
						fk_user_author, 
						fk_project, 
						label, 
						inventorycode, 
						fk_depot1, 
						fk_depot2, 
						date1, 
						date2, 
						shipper, 
						private_note, 
						pdf_note, 
						n_package,
						entity
					) VALUES (
						'".$user->id."', "
						.($this->fk_project ? "'".intval($this->fk_project)."'" : '0' ).", "
						.($this->label ? "'".$this->db->escape($this->label)."'" : "'Nueva transferencia'").", "
						.($this->inventorycode ? "'".$this->db->escape($this->inventorycode)."'" : 'NULL').", "
						.($this->fk_depot1 ? "'".intval($this->fk_depot1)."'" : '0' ).", "
						.($this->fk_depot2 ? "'".intval($this->fk_depot2)."'" : '0').", "
						.($this->date1 ? "'".$this->date1."'" : 'NULL').", "
						.($this->date2 ? "'".$this->date2."'" : 'NULL').", "
						.($this->shipper ? "'".$this->db->escape($this->shipper)."'" : 'NULL').", "
						.($this->private_note ? "'".$this->db->escape($this->private_note)."'" : 'NULL').", "
						.($this->pdf_note ? "'".$this->db->escape($this->pdf_note)."'" : 'NULL').", "
						.($this->n_package ? "'".$this->db->escape($this->n_package)."'" : 'NULL').", "
						."'".$conf->entity."'"
					.")";

		$this->db->begin();

		dol_syslog('@STSTST -- '.get_class($this)."::create sql=".$sql, LOG_DEBUG);

		// run SQL
		$resql=$this->db->query($sql);
		if (! $resql) { $error++; $this->errors[]="Error ".$this->db->lasterror(); }

		if (! $error) {
			$this->rowid = $this->db->last_insert_id(MAIN_DB_PREFIX."stockTransfers_transfers");
		}

		// Commit or rollback
		if ($error) {
			foreach ($this->errors as $errmsg) {
				dol_syslog('@STSTST -- '.get_class($this)."::create ERROR: ".$errmsg, LOG_ERR);
				$this->error.=($this->error?', '.$errmsg:$errmsg);
			}
			$this->db->rollback();
			return -1*$error;
		} else {
			$this->db->commit();
			return $this->id;
		}
	}

	/**
	 * Load a stocktransfer in memory from database
	 *
	 * @param	integer	$id Record ID at stocktransfers_transfers table
	 * @return	integer		<0 if KO, >0 if OK
	 */
	public function fetch($id)
	{
		global $langs;
		$sql = "SELECT * FROM ".MAIN_DB_PREFIX."stocktransfers_transfers";
		$sql.= " WHERE rowid = ".$id;
		dol_syslog('@STSTST -- '.get_class($this)."::fetch sql=".$sql, LOG_DEBUG);

		$resql=$this->db->query($sql);
		if ($resql) {
			if ($this->db->num_rows($resql)) {
				$row = $this->db->fetch_array($resql);
				if (is_array($row)) {
					foreach ($row as $f=>$v) $this->{$f} = $v;
				}
			}
			$this->db->free($resql);

			$this->unserializeProducts();

			return 1;
		} else {
			$this->error="Error ".$this->db->lasterror();
			dol_syslog('@STSTST -- '.get_class($this)."::fetch ERROR: ".$this->error, LOG_ERR);
			return -1;
		}
	}

	/**
	 * Save at databasse the loaded stocktransfer data
	 *
	 * @return	integer		<0 if KO, >0 if OK
	 */
	public function update()
	{
		global $conf, $langs;
		$error=0;

		// Check parameters
		if (empty($this->fk_depot1) || empty($this->fk_depot2)) {
			$this->error = "ErrorBadParameter";
			dol_syslog('@STSTST -- '.get_class($this)."::create ERROR: Try to create a transfer with an empty parameter (user, depots, ...)", LOG_ERR);
			return -3;
		}

		// Update request
		$sql = "UPDATE ".MAIN_DB_PREFIX."stocktransfers_transfers SET 
					label=".($this->label ? "'".$this->db->escape($this->label)."'" : "'Nueva transferencia'").", 
					fk_project=".($this->fk_project ? "'".intval($this->fk_project)."'" : '0' ).", 
					inventorycode=".($this->inventorycode ? "'".$this->db->escape($this->inventorycode)."'" : 'NULL').", 
					fk_depot1=".($this->fk_depot1 ? "'".intval($this->fk_depot1)."'" : '0' ).", 
					fk_depot2=".($this->fk_depot2 ? "'".intval($this->fk_depot2)."'" : '0' ).",";

		if (!empty($this->date1)) {
			$sql.= "date1=STR_TO_DATE('".str_replace('-', '', $this->date1)."','%Y%m%d'), "; // 20190613
		} else {
			$sql.= "date1=null, ";
		}

		if (!empty($this->date2)) {
			$sql.= "date2=STR_TO_DATE('".str_replace('-', '', $this->date2)."','%Y%m%d'), "; // 20190613
		} else {
			$sql.= "date2=null, ";
		}

		$sql.= "shipper=".($this->shipper ? "'".$this->db->escape($this->shipper)."'" : 'NULL').", 
				n_package=".($this->n_package ? "'".$this->db->escape($this->n_package)."'" : 'NULL').", 
				s_products='".$this->db->escape(is_array($this->products) ? serialize($this->products) : serialize(array()))."', 
				status='".$this->status."', 
				private_note='".$this->db->escape($this->private_note)."', 
				pdf_note='".$this->db->escape($this->pdf_note)."', 
				n_products='".count($this->products)."', 
				lang='".$this->db->escape($this->lang)."' 
			WHERE rowid=".$this->rowid;

		$this->db->begin();

		dol_syslog('@STSTST -- '.get_class($this)."::update sql=".$sql, LOG_DEBUG);

		// run query
		$resql = $this->db->query($sql);
		if (! $resql) { $error++; $this->errors[]="Error ".$this->db->lasterror(); }

		// Commit or rollback
		if ($error) {
			foreach ($this->errors as $errmsg) {
				dol_syslog('@STSTST -- '.get_class($this)."::update ERROR: ".$errmsg, LOG_ERR);
				$this->error.=($this->error?', '.$errmsg:$errmsg);
			}
			$this->db->rollback();
			return -1*$error;
		} else {
			$this->db->commit();
			return 1;
		}
	}


	/**
	 * Delete stocktransfer at database
	 *
	 * @return	integer		<0 if KO, >0 if OK
	 */
	public function delete()
	{

		global $conf, $langs;
		$error=0;

		$sql = "DELETE FROM ".MAIN_DB_PREFIX."stocktransfers_transfers WHERE rowid=".$this->rowid;

		$this->db->begin();

		dol_syslog('@STSTST -- '.get_class($this)."::delete sql=".$sql);

		$resql = $this->db->query($sql);
		if (! $resql) { $error++; $this->errors[]="Error ".$this->db->lasterror(); }

		// Commit or rollback
		if ($error) {
			foreach ($this->errors as $errmsg) {
				dol_syslog('@STSTST -- '.get_class($this)."::delete ERROR: ".$errmsg, LOG_ERR);
				$this->error.=($this->error?', '.$errmsg:$errmsg);
			}
			$this->db->rollback();
			return -1*$error;
		} else {
			$this->db->commit();
			return 1;
		}
	}

	/**
	 * Create stock movements OUT of depot1 or IN the depot2
	 *
	 * phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	 *
	 * @param	integer	$depot		'1' -> movements on depot1 / '2' -> movements on depot2
	 * @param	integer	$reverse	optional '1' -> add movements with negative quantity value
	 * @return	integer				<0 if KO, >0 if OK
	 */
	public function create_stock_movements($depot, $reverse = 0)
	{

		global $conf, $langs, $user;
		$error=0;

		$this->db->begin();

		$product = new Product($this->db);
		$id_sw = $this->fk_depot1;
		$id_tw = $this->fk_depot2;
		if ($depot=='1') {
			$fk_entrepot = $id_sw;
			$typemov = $reverse ? 0 : 1;
		} else {
			$fk_entrepot = $id_tw;
			$typemov = $reverse ? 1 : 0;
		}

		foreach ($this->products as $line_ii => $p) {
			$pid   = $p['id'];
			$batch = $p['b'];
			$qty   = floatval($p['n']);
			if ($qty==0) continue;

			if ($reverse && empty($p['m'.$depot])) continue; // we check that this product really has a stock movement ID, if not then we do nothing. It shouldn't be match never... so it's a redundant checking, by the way :)

			$dlc  ='';		// They are loaded later from serial
			$dluo ='';		// They are loaded later from serial

			if (! $error && is_numeric($qty) && $pid) {
				$result = $product->fetch($pid);

				$product->load_stock('novirtual');	// Load array product->stock_warehouse

				// Define value of products moved
				$pricesrc=0;
				if (! empty($product->pmp)) $pricesrc=$product->pmp;
				$pricedest=$pricesrc;

				$label = ($this->label).($reverse ? ' CANCEL':'');
				$invcode = ($this->inventorycode).($reverse ? ' CANCEL':'');

				if (!isModEnabled('productbatch') || ! $product->hasbatch()) {

					/* -- Add/Remove stock PRODUCT-WIHTOUT-BATCHS (If product does not need lot/serial) -- */
					$result = $product->correct_stock($user, $fk_entrepot, $qty, $typemov, $label, $pricesrc, $invcode);
					//_evar(json_encode(array($fk_entrepot, $qty, $typemov, $label, $pricesrc, $invcode)),'correct_stock');

				} else {
					/* -- Add/Remove stock PRODUCT-WIHT-BATCHS -- */
					$product_batchs_stock = $this->getProductLots($pid, $id_sw);
					$dlc  = '';
					$dluo = '';
					if (is_array($product_batchs_stock) && count($product_batchs_stock)>0
							&& isset($product_batchs_stock[$batch])) {
								$dlc  = !empty($product_batchs_stock[$batch]['eatby'])  ? strtotime($product_batchs_stock[$batch]['eatby'] .' 12:00:00') : '';
								$dluo = !empty($product_batchs_stock[$batch]['sellby']) ? strtotime($product_batchs_stock[$batch]['sellby'].' 12:00:00') : '';
					}

					$result = $product->correct_stock_batch($user, $fk_entrepot, $qty, $typemov, $label, $pricesrc, $dlc, $dluo, $batch, $invcode);
					//_evar(json_encode(array($fk_entrepot, $qty, $typemov, $label, $pricesrc, $dlc, $dluo, $batch, $invcode)),'correct_stock_batch');
				}

				if ($result < 0) {
					$error++;
					$_SESSION['EventMessages'][] = array($product->errors, null, 'errors');
				}
			} else {
				dol_print_error('', "Bad value saved into sessions");
				$error++;
			}
		}
		
		if (! $error) {
			// update the stock movements IDs stored at $transfer->products array
			if (!$reverse) {
				// get IDs of stock movements
				$sql = "SELECT * FROM ".MAIN_DB_PREFIX."stock_mouvement ORDER BY rowid DESC LIMIT ".count($this->products);
				$elements = f_query($sql, '');
				if (is_array($elements) && count($elements)>0) {
					foreach ($elements as $ele) {
						$pid     = $ele['fk_product'];
						$batch   = !empty($ele['batch']) ? trim($ele['batch']) : '';
						$batch_sane = trim($this->sanitize_batch($batch));
						$line_ii = $pid . ($batch_sane!='' ? '_'.$batch_sane : '');

						if ($ele['fk_entrepot']==$fk_entrepot && !empty($this->products[$line_ii])) {
								$this->products[$line_ii]['m'.$depot] = $ele['rowid'];
						}
					}
				}
			} else {
				// empty the IDs of the stock movements
				foreach ($this->products as $line_ii=>$p) {
					$this->products[$line_ii]['m'.$depot] = '';
				}
			}

			// to save IDs of the stock movements
			// it will let us to delete that movements later if it's requested by user
			$this->update();
			$this->db->commit();

			$_SESSION['EventMessages'][] = array($langs->trans("StockMovementRecorded"), null, 'mesgs');
			return 1;
		} else {
			$this->db->rollback();
			$_SESSION['EventMessages'][] = array($langs->trans("Error"), null, 'errors');
			return -1;
		}
	}

	/**
	 * Load an object from its id and create a new one in database
	 *
	 * @param	integer	$fromid	rowid of the record to be cloned
	 * @return	integer			rowid of the new record
	 */
	public function createFromClone($fromid)
	{
		global $user,$langs;

		$error=0;

		$object=new ImpBM_rule($this->db);

		$this->db->begin();

		// Load source object
		$object->fetch($fromid);
		$object->id=0;
		$object->statut=0;

		// Clear fields
		// ...

		// Create clone
		$result=$object->create($user);

		// Other options
		if ($result < 0) {
			$this->error=$object->error;
			$error++;
		}

		// End
		if (! $error) {
			$this->db->commit();
			return $object->id;
		} else {
			$this->db->rollback();
			return -1;
		}
	}

	/**
	 * Initialise object with example values
	 *
	 * @return void
	 */
	public function initAsSpecimen()
	{
		$this->id='';
		$this->pattern='';
		$this->ruleOrder='';
		$this->fk_account='';
		$this->category='';
	}

	/**
	 * Unserialize s_products
	 *
	 * @return void
	 */
	public function unserializeProducts()
	{
		$products = array();
		if (!empty($this->s_products)) {
			$arr = @unserialize($this->s_products);
			if (is_array($arr)) {
				$products = $arr;
			}
		}
		$this->products = $products;
		$this->n_products = count($products);
	}

	/**
	 * Return an array with current stock of products in the origin depot
	 *
	 * @return array
	 */
	public function getStock()
	{
		$stock = array();
		if ($this->fk_depot1 > 0 && count($this->products) > 0) {
			// extract array of product IDs
			$product_IDs = array();
			if (is_array($this->products)) {
				foreach ($this->products as $arr) $product_IDs[$arr['id']] = $arr['id'];
			}

			// get stock for those products
			$sql = "SELECT fk_product,fk_entrepot,reel 
						FROM ".MAIN_DB_PREFIX."product_stock 
						WHERE fk_product IN (".implode(',', array_keys($product_IDs)).")";

			$elements = f_query($sql, '');
			if (is_array($elements) && count($elements)>0) {
				foreach ($elements as $ele) {
					if (empty($ele['fk_entrepot']) || empty($ele['fk_product'])) continue;
					if (!isset($stock[$ele['fk_entrepot']])) $stock[$ele['fk_entrepot']] = array();
					$stock[$ele['fk_entrepot']][$ele['fk_product']] = $ele['reel'];
				}
			}
		}
		// _evard($stock,'$stock');
		return $stock;
	}

	/**
	 * Return an array with current stock of ONE product in BOTH WAREHOUSES
	 * and also an array of the stock of the different lots of this product (if there are) in the ORIGIN WAREHOUSE
	 *
	 * @param	integer	$product_id		Product ID
	 * @param	integer	$warehouse_id1	Origin warehouse ID
	 * @param	integer	$warehouse_id2	Destination warehouse ID
	 * @return	array					array($stock Array,$lots Array)
	 */
	public function getStockOneProduct($product_id, $warehouse_id1, $warehouse_id2)
	{
		$stock = array();
		$lots  = array();
		if (intval($product_id) > 0 && intval($warehouse_id1) > 0 && intval($warehouse_id2) > 0) {
			// global stock by warehouse
			$sql = "SELECT fk_product,reel,fk_entrepot 
						FROM ".MAIN_DB_PREFIX."product_stock 
						WHERE fk_entrepot IN ('".$warehouse_id1."','".$warehouse_id2."') 
						AND fk_product = ".$product_id;

			$elements = f_query($sql, '');
			if (is_array($elements) && count($elements)>0) {
				foreach ($elements as $ele) {
					if (empty($ele['fk_entrepot'])) continue;
					if ($ele['fk_entrepot']==$warehouse_id1) $stock['stock1'] = _qty($ele['reel']);
					if ($ele['fk_entrepot']==$warehouse_id2) $stock['stock2'] = _qty($ele['reel']);
				}
			}

			// stock by lot
			$lots  = $this->getProductLots($product_id, $warehouse_id1);
		}
		return array($stock,$lots);
	}

	/**
	 * Building an array indexed by batch/lot, to get a current stock number of each lot
	 *
	 * @param	integer	$product_id		Product ID
	 * @param	integer	$warehouse_id	Warehouse ID
	 * @return	array					Lots array
	 */
	public function getProductLots($product_id, $warehouse_id)
	{

		$product_id   = intval($product_id);
		$warehouse_id = intval($warehouse_id);

		require_once DOL_DOCUMENT_ROOT.'/product/class/productbatch.class.php';

		$lots = array();
		$sql = "SELECT rowid as product_stock_id FROM ".MAIN_DB_PREFIX."product_stock WHERE fk_product=".$product_id." AND fk_entrepot=".$warehouse_id;
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i=0;
			while ($i < $num) {
				$obj = $this->db->fetch_object($resql);
				$details = Productbatch::findAll($this->db, $obj->product_stock_id, 0, $product_id);
				if (count($details) > 0) {
					foreach ($details as $pdluo) {
						$batch = $pdluo->batch;
						if (!isset($lots[$batch])) $lots[$batch] = array('eatby'=>'', 'sellby'=>'', 'n'=>0);
						if ($lots[$batch]['eatby'] =='' && !is_null($pdluo->eatby)  && !empty($pdluo->eatby))  $lots[$batch]['eatby']  = date("Y-m-d", $pdluo->eatby);
						if ($lots[$batch]['sellby']=='' && !is_null($pdluo->sellby) && !empty($pdluo->sellby)) $lots[$batch]['sellby'] = date("Y-m-d", $pdluo->sellby);
						if (!is_null($pdluo->qty) && !empty($pdluo->qty)) $lots[$batch]['n'] = $lots[$batch]['n'] + intval($pdluo->qty);
					}
				}
				$i++;
			}
		}
		return $lots;
	}

	/**
	 * Building an array indexed by PRODUCT + batch/lot, to get a current stock number of each lot
	 * note: it's the same query above getProductLots() but with an array of product_id
	 *
	 * @param	array	$a_product_id	Array of products IDs
	 * @param	integer	$warehouse_id	Warehouse ID
	 * @return	array			$lots
	 */
	public function getProductsLots($a_product_id, $warehouse_id)
	{

		if (!is_array($a_product_id) || count($a_product_id)==0) return array();
		$warehouse_id = intval($warehouse_id);

		require_once DOL_DOCUMENT_ROOT.'/product/class/productbatch.class.php';

		$lots = array();
		$sql = "SELECT rowid as product_stock_id, fk_product FROM ".MAIN_DB_PREFIX."product_stock WHERE fk_product IN (".implode(',', $a_product_id).") AND fk_entrepot=".$warehouse_id;
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i=0;
			while ($i < $num) {
				$obj = $this->db->fetch_object($resql);
				$product_id = $obj->fk_product;
				$details = Productbatch::findAll($this->db, $obj->product_stock_id, 0, $product_id);
				if (count($details) > 0) {
					foreach ($details as $pdluo) {
						$batch = $pdluo->batch;
						if (!isset($lots[$product_id])) $lots[$product_id] = array();
						if (!isset($lots[$product_id][$batch])) $lots[$product_id][$batch] = array('eatby'=>'', 'sellby'=>'', 'n'=>0);
						if ($lots[$product_id][$batch]['eatby'] =='' && !is_null($pdluo->eatby)  && !empty($pdluo->eatby))  $lots[$product_id][$batch]['eatby']  = date("Y-m-d", $pdluo->eatby);
						if ($lots[$product_id][$batch]['sellby']=='' && !is_null($pdluo->sellby) && !empty($pdluo->sellby)) $lots[$product_id][$batch]['sellby'] = date("Y-m-d", $pdluo->sellby);
						if (!is_null($pdluo->qty) && !empty($pdluo->qty)) $lots[$product_id][$batch]['n'] = $lots[$product_id][$batch]['n'] + intval($pdluo->qty);
					}
				}
				$i++;
			}
		}
		return $lots;
	}

	/**
	 * Returns the last $max transfers
	 *
	 * @param	array	$vars	Parameters: 'max' number of elements
	 * @return	array			Array of last transfers or empty array
	 */
	public function getLatestTransfers($vars)
	{
		global $langs, $conf;

		$max = !empty($vars['max']) ? intval($vars['max']) : 5;

		$sql = "SELECT * FROM ".MAIN_DB_PREFIX."stocktransfers_transfers 
					ORDER BY rowid DESC 
					WHERE entity=".$conf->entity."
					LIMIT ".$max;

		dol_syslog('@STSTST -- '.get_class($this)."::fetch sql=".$sql, LOG_DEBUG);

		$elements = f_query($sql, '');
		if (is_array($elements)) return $elements;

		return array();
	}

	/**
	 * Clean non-ascii chars and spaces on batch_key, to be able to use it in URL, and HTML objects IDs, regexp, etc..
	 *
	 * phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	 *
	 *	@param      string	$str    Alphanumeric string to be cleaned
	 *	@return     string			Cleaned string
	 */
	public function sanitize_batch($str)
	{
		$str = str_replace(array('á','à','ä','â'), 'a', $str);
		$str = str_replace(array('Á','À','Ä','Â'), 'A', $str);
		$str = str_replace(array('é','è','ë','ê'), 'e', $str);
		$str = str_replace(array('É','È','Ë','Ê'), 'E', $str);
		$str = str_replace(array('í','ì','ï','î'), 'i', $str);
		$str = str_replace(array('Í','Ì','Ï','Î'), 'I', $str);
		$str = str_replace(array('ó','ò','ö','ô'), 'o', $str);
		$str = str_replace(array('Ó','Ò','Ö','Ô'), 'O', $str);
		$str = str_replace(array('ú','ù','ü','û'), 'u', $str);
		$str = str_replace(array('Ú','Ù','Ü','Ù'), 'U', $str);
		$str = str_replace(array('ñ','Ñ','ç','Ç','l·l','L·L'), array('n','N','s','S','l','L'), $str);
		$str = str_replace(array(' ',',',';',':','(',')','=','+','*','¿','?','[',']','{','}','&','%','$','#','@','|','~','/','\\'), '_', $str);
		$str = str_replace(array('"',"'",'¡','!','·','<','>'), '', $str);
		$str = str_replace(array('____','___','__'), '_', $str);
		$str = trim($str, '_');
		return $str;
	}
}
