<?php
// Copyright (C) 2023 TuNombre - Licencia GPLv3+

// Configuración de seguridad
if (!defined('NOSCRIPT')) define('NOSCRIPT', '1');
if (!defined('NOCSRFCHECK')) define('NOCSRFCHECK', '1');
if (!defined('NOTOKENRENEWAL')) define('NOTOKENRENEWAL', '1');

// Carga entorno Dolibarr
require '../../main.inc.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/admin.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/html.form.class.php';
require_once DOL_DOCUMENT_ROOT.'/societe/class/societe.class.php';
require_once DOL_DOCUMENT_ROOT.'/compta/facture/class/facture.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT.'/user/class/user.class.php';

// Habilitar modo de depuración
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Traducciones
$langs->loadLangs(array("admin", "bills", "companies", "ventasporcliente@ventasporcliente"));

// Verificar permisos
if (!$user->rights->societe->lire || (GETPOST('export') && !$user->rights->societe->export)) {
    accessforbidden();
}

// Obtener prefijo de la base de datos
$prefix = $conf->db->prefix;
if (empty($prefix)) {
    setEventMessages('El prefijo de la base de datos no está definido en conf.php. Configure $dolibarr_main_db_prefix correctamente.', null, 'errors');
    $prefix = 'llx_'; // Valor por defecto de respaldo
}

// Verificar existencia de tablas requeridas
$required_tables = array('facture', 'societe', 'facturedet', 'product', 'user');
foreach ($required_tables as $table) {
    $sql_check = "SHOW TABLES LIKE '{$prefix}{$table}'";
    $res_check = $db->query($sql_check);
    if (!$res_check || $db->num_rows($res_check) == 0) {
        setEventMessages("La tabla {$prefix}{$table} no existe en la base de datos. Verifique su instalación de Dolibarr.", null, 'errors');
    }
}

// Verificar si el datepicker de jQuery está habilitado
if (empty($conf->global->MAIN_USE_JQUERY_DATEPICKER)) {
    setEventMessages('El datepicker de jQuery está desactivado. Actívelo en Inicio > Configuración > Pantalla para una mejor experiencia.', null, 'warnings');
}

// Inicializar variables
$form = new Form($db);
$socid = GETPOST('socid', 'int');
$group_by = GETPOST('group_by', 'alpha') ? GETPOST('group_by', 'alpha') : 'factura';
$statut = GETPOST('statut', 'int') ? GETPOST('statut', 'int') : 0;

// Procesar fechas con validación
$date_start_ts = null;
$date_end_ts = null;

if (GETPOST('date_startday', 'int') && GETPOST('date_startmonth', 'int') && GETPOST('date_startyear', 'int')) {
    $start_hour = GETPOST('date_starthour', 'int') ?: 0;
    $start_minute = GETPOST('date_startminute', 'int') ?: 0;
    $start_day = GETPOST('date_startday', 'int');
    $start_month = GETPOST('date_startmonth', 'int');
    $start_year = GETPOST('date_startyear', 'int');
    
    if (checkdate($start_month, $start_day, $start_year)) {
        $date_start_ts = dol_mktime($start_hour, $start_minute, 0, $start_month, $start_day, $start_year);
    } else {
        setEventMessages($langs->trans("ErrorInvalidDate"), null, 'errors');
    }
} else {
    $date_start_ts = dol_time_plus_duree(dol_now(), -30, 'd');
}

if (GETPOST('date_endday', 'int') && GETPOST('date_endmonth', 'int') && GETPOST('date_endyear', 'int')) {
    $end_hour = GETPOST('date_endhour', 'int') ?: 23;
    $end_minute = GETPOST('date_endminute', 'int') ?: 59;
    $end_day = GETPOST('date_endday', 'int');
    $end_month = GETPOST('date_endmonth', 'int');
    $end_year = GETPOST('date_endyear', 'int');
    
    if (checkdate($end_month, $end_day, $end_year)) {
        $date_end_ts = dol_mktime($end_hour, $end_minute, 59, $end_month, $end_day, $end_year);
    } else {
        setEventMessages($langs->trans("ErrorInvalidDate"), null, 'errors');
    }
} else {
    $date_end_ts = dol_now();
}

// Depuración de los valores recibidos
if (defined('DEBUG') && DEBUG) {
    error_log("GETPOST values: date_startday=" . var_export(GETPOST('date_startday'), true) . 
              ", date_startmonth=" . var_export(GETPOST('date_startmonth'), true) . 
              ", date_startyear=" . var_export(GETPOST('date_startyear'), true) . 
              ", date_starthour=" . var_export(GETPOST('date_starthour'), true) . 
              ", date_startminute=" . var_export(GETPOST('date_startminute'), true));
}

/**
 * Función para escapar strings para JavaScript.
 *
 * @param string $str La cadena a escapar.
 * @return string La cadena escapada.
 */
function js_escape($str) {
    return str_replace(["\\", "'", "\n", "\r"], ["\\\\", "\\'", "\\n", "\\r"], $str);
}

/**
 * Función para exportar a Excel.
 *
 * @param DoliDB $db         Objeto de conexión a la base de datos.
 * @param Langs  $langs      Objeto de traducción.
 * @param int    $socid      ID de la empresa cliente.
 * @param int    $date_start Fecha de inicio en formato timestamp.
 * @param int    $date_end   Fecha de fin en formato timestamp.
 * @param string $group_by   Criterio de agrupación ('factura' o 'producto').
 * @param int    $statut     Estatus de la factura (0=todos, 1=validado, 2=pagado).
 * @param string $prefix     Prefijo de las tablas de la base de datos.
 * @return bool              Retorna true en caso de éxito, false en caso de error.
 */
function exportToExcel($db, $langs, $socid, $date_start, $date_end, $group_by = 'factura', $statut = 0, $prefix) {
    $sql = '';
    if ($group_by == 'factura') {
        $sql = "SELECT 
                    f.ref AS Factura,
                    f.datec AS Fecha_Creacion,
                    f.datef AS Fecha_Factura,
                    s.nom AS Cliente,
                    u.login AS Usuario,
                    CONCAT(u.firstname, ' ', u.lastname) AS UserFullName,
                    COALESCE(p.label, fd.description) AS Producto,
                    fd.qty AS Cantidad,
                    COALESCE(p.cost_price, 0) AS Precio_Compra_Unitario,
                    COALESCE(p.pmp, 0) AS Precio_Medio_Ponderado,
                    fd.subprice AS Precio_Venta_Unitario,
                    (fd.qty * COALESCE(p.cost_price, 0)) AS Total_Precio_Compra,
                    fd.subprice * fd.qty AS Total_Venta,
                    (fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) AS Margen_Ganancia_Linea
                FROM 
                    {$prefix}facture f
                INNER JOIN 
                    {$prefix}societe s ON f.fk_soc = s.rowid
                LEFT JOIN 
                    {$prefix}user u ON f.fk_user_author = u.rowid
                LEFT JOIN 
                    {$prefix}facturedet fd ON f.rowid = fd.fk_facture
                LEFT JOIN 
                    {$prefix}product p ON fd.fk_product = p.rowid
                WHERE 
                    f.fk_soc = ".((int)$socid)." 
                    AND f.datec BETWEEN '".$db->idate($date_start)."' AND '".$db->idate($date_end)."' 
                    AND f.fk_statut ".($statut > 0 ? "= ".((int)$statut) : "> 0")."
                    AND f.type IN (0, 1)
                ORDER BY 
                    f.datec ASC, f.ref, fd.rowid";
    } elseif ($group_by == 'producto') {
        $sql = "SELECT 
                    COALESCE(p.label, fd.description) AS Producto,
                    SUM(fd.qty) AS Cantidad_Total,
                    AVG(fd.subprice) AS Precio_Venta_Unitario_Promedio,
                    AVG(COALESCE(p.cost_price, 0)) AS Precio_Compra_Unitario_Promedio,
                    AVG(COALESCE(p.pmp, 0)) AS Precio_Medio_Ponderado,
                    SUM(fd.qty * COALESCE(p.cost_price, 0)) AS Total_Precio_Compra,
                    SUM(fd.total_ht) AS Total_Venta,
                    SUM(fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) AS Margen_Ganancia_Total,
                    CASE 
                        WHEN SUM(fd.total_ht) > 0 
                        THEN (SUM(fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) / SUM(fd.total_ht) * 100) 
                        ELSE 0 
                    END AS Porcentaje_Margen
                FROM 
                    {$prefix}facture f
                INNER JOIN 
                    {$prefix}societe s ON f.fk_soc = s.rowid
                LEFT JOIN 
                    {$prefix}facturedet fd ON f.rowid = fd.fk_facture
                LEFT JOIN 
                    {$prefix}product p ON fd.fk_product = p.rowid
                WHERE 
                    f.fk_soc = ".((int)$socid)." 
                    AND f.datec BETWEEN '".$db->idate($date_start)."' AND '".$db->idate($date_end)."' 
                    AND f.fk_statut ".($statut > 0 ? "= ".((int)$statut) : "> 0")."
                    AND f.type IN (0, 1)
                GROUP BY 
                    COALESCE(p.label, fd.description)
                ORDER BY 
                    Producto ASC";
    }

    $resql = $db->query($sql);
    
    if (!$resql) {
        setEventMessages($langs->trans("ErrorSQL")." : ".$db->lasterror(), null, 'errors');
        return false;
    }

    // Establecer cabeceras para forzar la descarga del archivo Excel
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="ventas_cliente_'.$socid.'_'.date('Ymd_His').'.xls"');
    header('Cache-Control: max-age=0');
    header('Expires: 0');
    header('Pragma: public');
    
    echo "<html>
          <head>
          <meta http-equiv='content-type' content='text/plain; charset=UTF-8'/>
          </head>
          <body>
          <table border='1'>";
    
    // Imprimir encabezados de la tabla
    if ($group_by == 'factura') {
        echo "<tr>
                <th>".$langs->trans("Invoice")."</th>
                <th>".$langs->trans("DateCreation")."</th>
                <th>".$langs->trans("DateInvoice")."</th>
                <th>".$langs->trans("Customer")."</th>
                <th>".$langs->trans("User")."</th>
                <th>".$langs->trans("UserFullName")."</th>
                <th>".$langs->trans("Product")."</th>
                <th>".$langs->trans("Quantity")."</th>
                <th>".$langs->trans("PrecioCompraUnitario")."</th>
                <th>".$langs->trans("PrecioMedioPonderado")."</th>
                <th>".$langs->trans("PrecioVentaUnitario")."</th>
                <th>".$langs->trans("TotalPrecioCompra")."</th>
                <th>".$langs->trans("TotalVenta")."</th>
                <th>".$langs->trans("MargenGananciaLine")."</th>
              </tr>";
    } elseif ($group_by == 'producto') {
        echo "<tr>
                <th>".$langs->trans("Product")."</th>
                <th>".$langs->trans("CantidadTotal")."</th>
                <th>".$langs->trans("PrecioVentaUnitarioPromedio")."</th>
                <th>".$langs->trans("PrecioCompraUnitarioPromedio")."</th>
                <th>".$langs->trans("PrecioMedioPonderado")."</th>
                <th>".$langs->trans("TotalPrecioCompra")."</th>
                <th>".$langs->trans("TotalVenta")."</th>
                <th>".$langs->trans("MargenGananciaTotal")."</th>
                <th>".$langs->trans("PorcentajeMargen")."</th>
              </tr>";
    }
    
    // Imprimir datos y calcular totales
    $total_venta = 0;
    $total_precio_compra = 0;
    $total_pmp = 0;
    $total_margen = 0;
    $num_rows = $db->num_rows($resql);
    if ($num_rows == 0) {
        echo "<tr><td colspan='".($group_by == 'factura' ? 14 : 9)."'>".$langs->trans("NoData")."</td></tr>";
    } else {
        while ($obj = $db->fetch_object($resql)) {
            if ($group_by == 'factura') {
                echo "<tr>
                        <td>".($obj->Factura ? $obj->Factura : '-')."</td>
                        <td>".dol_print_date($db->jdate($obj->Fecha_Creacion), 'dayhour')."</td>
                        <td>".dol_print_date($db->jdate($obj->Fecha_Factura), 'day')."</td>
                        <td>".($obj->Cliente ? $obj->Cliente : '-')."</td>
                        <td>".($obj->Usuario ? $obj->Usuario : '-')."</td>
                        <td>".($obj->UserFullName ? $obj->UserFullName : '-')."</td>
                        <td>".($obj->Producto ? $obj->Producto : '-')."</td>
                        <td>".($obj->Cantidad ? $obj->Cantidad : 0)."</td>
                        <td>".price($obj->Precio_Compra_Unitario)."</td>
                        <td>".price($obj->Precio_Medio_Ponderado)."</td>
                        <td>".price($obj->Precio_Venta_Unitario)."</td>
                        <td>".price($obj->Total_Precio_Compra)."</td>
                        <td>".price($obj->Total_Venta)."</td>
                        <td>".price($obj->Margen_Ganancia_Linea)."</td>
                      </tr>";
                
                $total_venta += floatval($obj->Total_Venta);
                $total_precio_compra += floatval($obj->Total_Precio_Compra);
                $total_pmp += floatval($obj->Precio_Medio_Ponderado);
                $total_margen += floatval($obj->Margen_Ganancia_Linea);
            } elseif ($group_by == 'producto') {
                echo "<tr>
                        <td>".($obj->Producto ? $obj->Producto : '-')."</td>
                        <td>".$obj->Cantidad_Total."</td>
                        <td>".price($obj->Precio_Venta_Unitario_Promedio)."</td>
                        <td>".price($obj->Precio_Compra_Unitario_Promedio)."</td>
                        <td>".price($obj->Precio_Medio_Ponderado)."</td>
                        <td>".price($obj->Total_Precio_Compra)."</td>
                        <td>".price($obj->Total_Venta)."</td>
                        <td>".price($obj->Margen_Ganancia_Total)."</td>
                        <td>".price($obj->Porcentaje_Margen, 2)."%</td>
                      </tr>";
                
                $total_venta += floatval($obj->Total_Venta);
                $total_precio_compra += floatval($obj->Total_Precio_Compra);
                $total_margen += floatval($obj->Margen_Ganancia_Total);
            }
        }
    }
    
    // Imprimir fila de totales
    if ($group_by == 'factura') {
        echo "<tr>
                <td colspan='11'><b>".$langs->trans("Total")."</b></td>
                <td><b>".price($total_precio_compra)."</b></td>
                <td><b>".price($total_venta)."</b></td>
                <td><b>".price($total_margen)."</b></td>
              </tr>";
    } elseif ($group_by == 'producto') {
        echo "<tr>
                <td colspan='5'><b>".$langs->trans("Total")."</b></td>
                <td><b>".price($total_precio_compra)."</b></td>
                <td><b>".price($total_venta)."</b></td>
                <td><b>".price($total_margen)."</b></td>
                <td></td>
              </tr>";
    }
    
    echo "</table></body></html>";
    exit;
}

// Procesar exportación
if (GETPOST('export') && $socid > 0 && $date_start_ts > 0 && $date_end_ts > 0 && $date_start_ts <= $date_end_ts) {
    if (!exportToExcel($db, $langs, $socid, $date_start_ts, $date_end_ts, $group_by, $statut, $prefix)) {
        setEventMessages($langs->trans("ErrorExportFailed"), null, 'errors');
    }
}

// Configuración página
$title = $langs->trans("VentasPorClienteReport");
llxHeader('', $title, '', '', 0, 0, array('/includes/jquery/plugins/datepicker3/jquery-ui.min.js', '/includes/jquery/plugins/datepicker3/i18n/datepicker-es.js'), array('/includes/jquery/plugins/datepicker3/jquery-ui.css'));

print load_fiche_titre($title, '', 'bill');

// Formulario de búsqueda
print '<form method="POST" action="'.$_SERVER['PHP_SELF'].'" id="searchForm">';
print '<input type="hidden" name="token" value="'.newToken().'">';
print '<table class="noborder centpercent">';
print '<tr class="liste_titre">';
print '<td>'.$langs->trans("Customer").'</td>';
print '<td>'.$langs->trans("DateStart").'</td>';
print '<td>'.$langs->trans("DateEnd").'</td>';
print '<td>'.$langs->trans("Status").'</td>';
print '<td>'.$langs->trans("GroupBy").'</td>';
print '<td colspan="2"></td>';
print '</tr>';

print '<tr class="oddeven">';
print '<td>'.$form->select_company($socid, 'socid', '', 1).'</td>';
print '<td>';
print $form->selectDate($date_start_ts, 'date_start', 1, 1, 0, '', 1, 1, 0, 'showOn: "both", buttonImage: "'.DOL_URL_ROOT.'/theme/eldy/img/object_calendarday.png", buttonImageOnly: true, changeMonth: true, changeYear: true');
print '</td>';
print '<td>';
print $form->selectDate($date_end_ts, 'date_end', 1, 1, 0, '', 1, 1, 0, 'showOn: "both", buttonImage: "'.DOL_URL_ROOT.'/theme/eldy/img/object_calendarday.png", buttonImageOnly: true, changeMonth: true, changeYear: true');
print '</td>';
print '<td>';
print $form->selectarray('statut', array(0 => $langs->trans("All"), 1 => $langs->trans("Validated"), 2 => $langs->trans("Paid")), $statut);
print '</td>';
print '<td>';
print $form->selectarray('group_by', array('factura' => $langs->trans("ByInvoice"), 'producto' => $langs->trans("ByProduct")), $group_by);
print '</td>';
print '<td class="center"><input type="submit" class="button button-search" name="search" value="'.$langs->trans("Search").'"></td>';
print '<td class="center"><input type="submit" class="button button-export" name="export" value="'.$langs->trans("Export").'"></td>';
print '</tr>';
print '</table>';
print '</form>';

// JavaScript para inicializar el datepicker
print '<script>
    jQuery.noConflict();
    jQuery(document).ready(function($) {
        $.datepicker.setDefaults($.datepicker.regional["es"]);
        $("#date_start").datepicker({
            dateFormat: "dd/mm/yy",
            changeMonth: true,
            changeYear: true,
            showOn: "both",
            buttonImage: "'.DOL_URL_ROOT.'/theme/eldy/img/object_calendarday.png",
            buttonImageOnly: true,
            showButtonPanel: true
        });
        $("#date_end").datepicker({
            dateFormat: "dd/mm/yy",
            changeMonth: true,
            changeYear: true,
            showOn: "both",
            buttonImage: "'.DOL_URL_ROOT.'/theme/eldy/img/object_calendarday.png",
            buttonImageOnly: true,
            showButtonPanel: true
        });
    });
</script>';

// Validar formulario
if (GETPOST('search') && $socid <= 0) {
    setEventMessages($langs->trans("ErrorSelectCustomer"), null, 'errors');
}

// Mostrar resultados
if ($socid > 0 && $date_start_ts > 0 && $date_end_ts > 0 && GETPOST('search')) {
    $sql = '';
    if ($group_by == 'factura') {
        $sql = "SELECT 
                    f.ref AS Factura,
                    f.datec AS Fecha_Creacion,
                    f.datef AS Fecha_Factura,
                    s.nom AS Cliente,
                    u.login AS Usuario,
                    CONCAT(u.firstname, ' ', u.lastname) AS UserFullName,
                    COALESCE(p.label, fd.description) AS Producto,
                    fd.qty AS Cantidad,
                    COALESCE(p.cost_price, 0) AS Precio_Compra_Unitario,
                    COALESCE(p.pmp, 0) AS Precio_Medio_Ponderado,
                    fd.subprice AS Precio_Venta_Unitario,
                    (fd.qty * COALESCE(p.cost_price, 0)) AS Total_Precio_Compra,
                    fd.subprice * fd.qty AS Total_Venta,
                    (fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) AS Margen_Ganancia_Linea
                FROM 
                    {$prefix}facture f
                INNER JOIN 
                    {$prefix}societe s ON f.fk_soc = s.rowid
                LEFT JOIN 
                    {$prefix}user u ON f.fk_user_author = u.rowid
                LEFT JOIN 
                    {$prefix}facturedet fd ON f.rowid = fd.fk_facture
                LEFT JOIN 
                    {$prefix}product p ON fd.fk_product = p.rowid
                WHERE 
                    f.fk_soc = ".((int)$socid)." 
                    AND f.datec BETWEEN '".$db->idate($date_start_ts)."' AND '".$db->idate($date_end_ts)."' 
                    AND f.fk_statut ".($statut > 0 ? "= ".((int)$statut) : "> 0")."
                    AND f.type IN (0, 1)
                ORDER BY 
                    f.datec ASC, f.ref, fd.rowid";
    } elseif ($group_by == 'producto') {
        $sql = "SELECT 
                    COALESCE(p.label, fd.description) AS Producto,
                    SUM(fd.qty) AS Cantidad_Total,
                    AVG(fd.subprice) AS Precio_Venta_Unitario_Promedio,
                    AVG(COALESCE(p.cost_price, 0)) AS Precio_Compra_Unitario_Promedio,
                    AVG(COALESCE(p.pmp, 0)) AS Precio_Medio_Ponderado,
                    SUM(fd.qty * COALESCE(p.cost_price, 0)) AS Total_Precio_Compra,
                    SUM(fd.total_ht) AS Total_Venta,
                    SUM(fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) AS Margen_Ganancia_Total,
                    CASE 
                        WHEN SUM(fd.total_ht) > 0 
                        THEN (SUM(fd.total_ht - (fd.qty * COALESCE(p.cost_price, 0))) / SUM(fd.total_ht) * 100) 
                        ELSE 0 
                    END AS Porcentaje_Margen
                FROM 
                    {$prefix}facture f
                INNER JOIN 
                    {$prefix}societe s ON f.fk_soc = s.rowid
                LEFT JOIN 
                    {$prefix}facturedet fd ON f.rowid = fd.fk_facture
                LEFT JOIN 
                    {$prefix}product p ON fd.fk_product = p.rowid
                WHERE 
                    f.fk_soc = ".((int)$socid)." 
                    AND f.datec BETWEEN '".$db->idate($date_start_ts)."' AND '".$db->idate($date_end_ts)."' 
                    AND f.fk_statut ".($statut > 0 ? "= ".((int)$statut) : "> 0")."
                    AND f.type IN (0, 1)
                GROUP BY 
                    COALESCE(p.label, fd.description)
                ORDER BY 
                    Producto ASC";
    }

    $resql = $db->query($sql);

    if ($resql) {
        $num_rows = $db->num_rows($resql);
        print '<div class="div-table-responsive">';
        print '<table class="liste centpercent">';
        print '<tr class="liste_titre">';
        if ($group_by == 'factura') {
            print '<td>'.$langs->trans("Invoice").'</td>';
            print '<td>'.$langs->trans("DateCreation").'</td>';
            print '<td>'.$langs->trans("DateInvoice").'</td>';
            print '<td>'.$langs->trans("Customer").'</td>';
            print '<td>'.$langs->trans("User").'</td>';
            print '<td>'.$langs->trans("UserFullName").'</td>';
            print '<td>'.$langs->trans("Product").'</td>';
            print '<td class="right">'.$langs->trans("Quantity").'</td>';
            print '<td class="right">'.$langs->trans("PrecioCompraUnitario").'</td>';
            print '<td class="right">'.$langs->trans("PrecioMedioPonderado").'</td>';
            print '<td class="right">'.$langs->trans("PrecioVentaUnitario").'</td>';
            print '<td class="right">'.$langs->trans("TotalPrecioCompra").'</td>';
            print '<td class="right">'.$langs->trans("TotalVenta").'</td>';
            print '<td class="right">'.$langs->trans("MargenGananciaLine").'</td>';
        } elseif ($group_by == 'producto') {
            print '<td>'.$langs->trans("Product").'</td>';
            print '<td class="right">'.$langs->trans("CantidadTotal").'</td>';
            print '<td class="right">'.$langs->trans("PrecioVentaUnitarioPromedio").'</td>';
            print '<td class="right">'.$langs->trans("PrecioCompraUnitarioPromedio").'</td>';
            print '<td class="right">'.$langs->trans("PrecioMedioPonderado").'</td>';
            print '<td class="right">'.$langs->trans("TotalPrecioCompra").'</td>';
            print '<td class="right">'.$langs->trans("TotalVenta").'</td>';
            print '<td class="right">'.$langs->trans("MargenGananciaTotal").'</td>';
            print '<td class="right">'.$langs->trans("PorcentajeMargen").'</td>';
        }
        print '</tr>';

        $total_venta = 0;
        $total_precio_compra = 0;
        $total_pmp = 0;
        $total_margen = 0;
        if ($num_rows == 0) {
            print '<tr><td colspan="'.($group_by == 'factura' ? 14 : 9).'">'.$langs->trans("NoData").'</td></tr>';
        } else {
            while ($obj = $db->fetch_object($resql)) {
                print '<tr class="oddeven">';
                if ($group_by == 'factura') {
                    print '<td>'.($obj->Factura ? $obj->Factura : '-').'</td>';
                    print '<td>'.dol_print_date($db->jdate($obj->Fecha_Creacion), 'dayhour').'</td>';
                    print '<td>'.dol_print_date($db->jdate($obj->Fecha_Factura), 'day').'</td>';
                    print '<td>'.($obj->Cliente ? $obj->Cliente : '-').'</td>';
                    print '<td>'.($obj->Usuario ? $obj->Usuario : '-').'</td>';
                    print '<td>'.($obj->UserFullName ? $obj->UserFullName : '-').'</td>';
                    print '<td>'.($obj->Producto ? $obj->Producto : '-').'</td>';
                    print '<td class="right">'.($obj->Cantidad ? $obj->Cantidad : 0).'</td>';
                    print '<td class="right">'.price($obj->Precio_Compra_Unitario).'</td>';
                    print '<td class="right">'.price($obj->Precio_Medio_Ponderado).'</td>';
                    print '<td class="right">'.price($obj->Precio_Venta_Unitario).'</td>';
                    print '<td class="right">'.price($obj->Total_Precio_Compra).'</td>';
                    print '<td class="right">'.price($obj->Total_Venta).'</td>';
                    print '<td class="right">'.price($obj->Margen_Ganancia_Linea).'</td>';
                    
                    $total_venta += floatval($obj->Total_Venta);
                    $total_precio_compra += floatval($obj->Total_Precio_Compra);
                    $total_pmp += floatval($obj->Precio_Medio_Ponderado);
                    $total_margen += floatval($obj->Margen_Ganancia_Linea);
                } elseif ($group_by == 'producto') {
                    print '<td>'.($obj->Producto ? $obj->Producto : '-').'</td>';
                    print '<td class="right">'.$obj->Cantidad_Total.'</td>';
                    print '<td class="right">'.price($obj->Precio_Venta_Unitario_Promedio).'</td>';
                    print '<td class="right">'.price($obj->Precio_Compra_Unitario_Promedio).'</td>';
                    print '<td class="right">'.price($obj->Precio_Medio_Ponderado).'</td>';
                    print '<td class="right">'.price($obj->Total_Precio_Compra).'</td>';
                    print '<td class="right">'.price($obj->Total_Venta).'</td>';
                    print '<td class="right">'.price($obj->Margen_Ganancia_Total).'</td>';
                    print '<td class="right">'.price($obj->Porcentaje_Margen, 2).'%</td>';
                    
                    $total_venta += floatval($obj->Total_Venta);
                    $total_precio_compra += floatval($obj->Total_Precio_Compra);
                    $total_margen += floatval($obj->Margen_Ganancia_Total);
                }
                print '</tr>';
            }
        }

        print '<tr class="liste_total">';
        if ($group_by == 'factura') {
            print '<td colspan="11">'.$langs->trans("Total").'</td>';
            print '<td class="right">'.price($total_precio_compra).'</td>';
            print '<td class="right">'.price($total_venta).'</td>';
            print '<td class="right">'.price($total_margen).'</td>';
        } elseif ($group_by == 'producto') {
            print '<td colspan="5">'.$langs->trans("Total").'</td>';
            print '<td class="right">'.price($total_precio_compra).'</td>';
            print '<td class="right">'.price($total_venta).'</td>';
            print '<td class="right">'.price($total_margen).'</td>';
            print '<td class="right"></td>';
        }
        print '</tr>';

        print '</table>';

        print '</div>';

        // Gráfico
        print '<h3>'.$langs->trans("SalesAndMarginByProduct").'</h3>';
        $sql_chart = "SELECT 
                            COALESCE(p.label, fd.description) AS Producto,
                            SUM(fd.qty) AS Cantidad_Total,
                            SUM(fd.total_ht) AS Total_Venta,
                            SUM(fd.qty * COALESCE(p.cost_price, 0)) AS Total_Precio_Compra
                        FROM 
                            {$prefix}facture f
                        INNER JOIN 
                            {$prefix}facturedet fd ON f.rowid = fd.fk_facture
                        LEFT JOIN 
                            {$prefix}product p ON fd.fk_product = p.rowid
                        WHERE 
                            f.fk_soc = ".((int)$socid)." 
                            AND f.datec BETWEEN '".$db->idate($date_start_ts)."' AND '".$db->idate($date_end_ts)."' 
                            AND f.fk_statut ".($statut > 0 ? "= ".((int)$statut) : "> 0")."
                            AND f.type IN (0, 1)
                        GROUP BY 
                            Producto
                        ORDER BY 
                            Producto ASC";
        $resql_chart = $db->query($sql_chart);
        if ($resql_chart) {
            print '<canvas id="salesChart" width="400" height="200"></canvas>';
            print '<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>';
            print '<script>
                        jQuery.noConflict();
                        jQuery(document).ready(function($) {
                            const ctx = document.getElementById("salesChart").getContext("2d");
                            const productos = [];
                            const totalCompra = [];
                            const totalVenta = [];
                            const margenGanancia = [];
                            const cantidadPiezas = [];
                            
                            ';
            while ($obj_chart = $db->fetch_object($resql_chart)) {
                $margen = $obj_chart->Total_Venta - $obj_chart->Total_Precio_Compra;
                echo "productos.push('".js_escape($obj_chart->Producto)."');\n";
                echo "totalCompra.push(".floatval($obj_chart->Total_Precio_Compra).");\n";
                echo "totalVenta.push(".floatval($obj_chart->Total_Venta).");\n";
                echo "margenGanancia.push(".floatval($margen).");\n";
                echo "cantidadPiezas.push(".floatval($obj_chart->Cantidad_Total).");\n";
            }
            print '
                            const chartData = {
                                labels: productos,
                                datasets: [
                                    {
                                        label: "Total Precio Compra",
                                        data: totalCompra,
                                        backgroundColor: "rgba(255, 159, 64, 0.8)",
                                        borderColor: "rgba(255, 159, 64, 1)",
                                        borderWidth: 1
                                    },
                                    {
                                        label: "Total Venta",
                                        data: totalVenta,
                                        backgroundColor: "rgba(75, 192, 192, 0.8)",
                                        borderColor: "rgba(75, 192, 192, 1)",
                                        borderWidth: 1
                                    },
                                    {
                                        label: "Margen de Ganancia",
                                        data: margenGanancia,
                                        backgroundColor: "rgba(153, 102, 255, 0.8)",
                                        borderColor: "rgba(153, 102, 255, 1)",
                                        borderWidth: 1
                                    }
                                ]
                            };

                            const chartOptions = {
                                responsive: true,
                                scales: {
                                    x: {
                                        stacked: false,
                                        title: {
                                            display: true,
                                            text: "Producto"
                                        }
                                    },
                                    y: {
                                        beginAtZero: true,
                                        stacked: false,
                                        title: {
                                            display: true,
                                            text: "Monto"
                                        }
                                    }
                                },
                                plugins: {
                                    tooltip: {
                                        callbacks: {
                                            title: function(context) {
                                                const index = context[0].dataIndex;
                                                return "Producto: " + productos[index];
                                            },
                                            label: function(context) {
                                                let label = context.dataset.label || "";
                                                if (label) {
                                                    label += ": ";
                                                }
                                                if (context.parsed.y !== null) {
                                                    label += new Intl.NumberFormat("es-MX", { style: "currency", currency: "MXN" }).format(context.parsed.y);
                                                }
                                                return label;
                                            },
                                            afterBody: function(context) {
                                                const index = context[0].dataIndex;
                                                return "Cantidad de piezas: " + cantidadPiezas[index];
                                            }
                                        }
                                    }
                                }
                            };

                            new Chart(ctx, {
                                type: "bar",
                                data: chartData,
                                options: chartOptions
                            });
                        });
                    </script>';
        } else {
            setEventMessages($langs->trans("ErrorSQLChart")." : ".$db->lasterror(), null, 'errors');
        }
    } else {
        setEventMessages($langs->trans("ErrorSQL")." : ".$db->lasterror(), null, 'errors');
    }
} elseif (GETPOST('search') && (!$date_start_ts || !$date_end_ts || $socid <= 0)) {
    print '<p>'.$langs->trans("NoData").'</p>';
}

llxFooter();
$db->close();
?>